import { jsxs, jsx } from "react/jsx-runtime";
import { useNavigate, Link } from "@tanstack/react-router";
import { useQueryClient, useQuery, useMutation } from "@tanstack/react-query";
import { format } from "date-fns";
import { es } from "date-fns/locale";
import { useMemo, useState, useEffect } from "react";
import { HugeiconsIcon } from "@hugeicons/react";
import { File02Icon, ArrowLeft01Icon, CheckmarkCircle02Icon, Comment01Icon, Tick02Icon } from "@hugeicons/core-free-icons";
import { B as Badge } from "./badge-D2nuHzhh.mjs";
import { B as Button } from "./button-Bp95UwA9.mjs";
import { C as Card, a as CardHeader, b as CardTitle, c as CardDescription, d as CardContent } from "./card-B_lDvuvx.mjs";
import { L as Label } from "./label-Cp5Wz1sX.mjs";
import { S as Select, a as SelectTrigger, b as SelectValue, c as SelectContent, d as SelectItem } from "./select-D2Xe1kOZ.mjs";
import { S as Spinner } from "./spinner-jf2ZHWGj.mjs";
import { T as Table, a as TableHeader, b as TableRow, c as TableHead, d as TableBody, e as TableCell } from "./table-SNBC4Iim.mjs";
import { T as Textarea } from "./textarea-D2EGKnT3.mjs";
import { R as Route$6, g as getUsuarioSesion, b as apiUrl } from "./router-5yTRMCgc.mjs";
import "class-variance-authority";
import "radix-ui";
import "./utils-H80jjgLf.mjs";
import "clsx";
import "tailwind-merge";
import "next-themes";
import "sonner";
const defaults = {
  credentials: "include",
  headers: { "Content-Type": "application/json" }
};
async function listarProyectosRevision(comite) {
  const query = comite != null ? `?comite=${comite}` : "";
  const res = await fetch(apiUrl(`/api/revision/proyectos${query}`), { ...defaults, method: "GET" });
  const data = await res.json().catch(() => ({}));
  if (!res.ok) return { success: false, error: data?.error || "Error al listar proyectos" };
  return {
    success: true,
    comite: data.comite,
    proyectos: data.proyectos || []
  };
}
async function obtenerProyectoRevision(id, comite) {
  const query = comite != null ? `?comite=${comite}` : "";
  const res = await fetch(apiUrl(`/api/revision/proyectos/${id}${query}`), { ...defaults, method: "GET" });
  const data = await res.json().catch(() => ({}));
  if (!res.ok) return { success: false, error: data?.error || "Error al cargar proyecto" };
  return {
    success: true,
    comite: data.comite,
    pasosPermitidos: data.pasosPermitidos || [],
    nombresPaso: data.nombresPaso || [],
    proyecto: data.proyecto,
    datosPorPaso: data.datosPorPaso || {},
    archivos: data.archivos || [],
    comentarios: data.comentarios || [],
    aprobaciones: data.aprobaciones || []
  };
}
async function crearComentarioRevision(proyectoId, body, comite) {
  const query = comite != null ? `?comite=${comite}` : "";
  const res = await fetch(apiUrl(`/api/revision/proyectos/${proyectoId}/comentarios${query}`), {
    ...defaults,
    method: "POST",
    body: JSON.stringify(body)
  });
  const data = await res.json().catch(() => ({}));
  if (!res.ok) return { success: false, error: data?.error || "No se pudo guardar comentario" };
  return { success: true, id: data.id };
}
async function aprobarProyectoRevision(proyectoId, comite) {
  const query = comite != null ? `?comite=${comite}` : "";
  const res = await fetch(apiUrl(`/api/revision/proyectos/${proyectoId}/aprobar${query}`), {
    ...defaults,
    method: "POST"
  });
  const data = await res.json().catch(() => ({}));
  if (!res.ok) return { success: false, error: data?.error || "No se pudo aprobar el proyecto" };
  return { success: true, aprobadoFinal: Boolean(data.aprobadoFinal) };
}
function urlArchivoRevision(proyectoId, tipoArchivo, comite) {
  const query = comite != null ? `?comite=${comite}` : "";
  return apiUrl(`/api/revision/proyectos/${proyectoId}/archivos/${tipoArchivo}${query}`);
}
const TIPOS_ARCHIVO_POR_PASO = {
  0: ["carta_aprobacion"],
  1: ["carta_colaboracion"],
  5: ["marco_referencia"],
  6: ["recoleccion_datos"],
  7: ["programa_gasto", "cronograma"]
};
function RevisionPage() {
  const queryClient = useQueryClient();
  const navigate = useNavigate();
  const search = Route$6.useSearch();
  const usuario = getUsuarioSesion();
  const esAdmin = useMemo(() => {
    const rol = String(usuario?.rol_nombre || "").normalize("NFD").replace(/[\u0300-\u036f]/g, "").toLowerCase();
    return rol.includes("admin");
  }, [usuario?.rol_nombre]);
  const [comiteActivo, setComiteActivo] = useState(search.comite ?? 1);
  const [comentariosDraft, setComentariosDraft] = useState({});
  const [pasoGuardando, setPasoGuardando] = useState(null);
  useEffect(() => {
    if (search.comite && search.comite !== comiteActivo) {
      setComiteActivo(search.comite);
    }
  }, [search.comite, comiteActivo]);
  const proyectoSeleccionadoId = search.proyecto ?? null;
  const queryLista = useQuery({
    queryKey: ["revision", "proyectos", esAdmin ? comiteActivo : null],
    queryFn: async () => {
      const r = await listarProyectosRevision(esAdmin ? comiteActivo : void 0);
      if (!r.success) throw new Error(r.error || "Error al cargar revisión");
      return r;
    }
  });
  const queryDetalle = useQuery({
    queryKey: ["revision", "detalle", proyectoSeleccionadoId, esAdmin ? comiteActivo : null],
    queryFn: async () => {
      if (proyectoSeleccionadoId == null) throw new Error("Proyecto no seleccionado");
      const r = await obtenerProyectoRevision(proyectoSeleccionadoId, esAdmin ? comiteActivo : void 0);
      if (!r.success) throw new Error(r.error || "Error al cargar el proyecto");
      return r;
    },
    enabled: proyectoSeleccionadoId != null
  });
  const mutationComentario = useMutation({
    mutationFn: async (input) => {
      const r = await crearComentarioRevision(input.proyectoId, {
        paso: input.paso,
        comentario: input.comentario
      }, esAdmin ? comiteActivo : void 0);
      if (!r.success) throw new Error(r.error || "No se pudo guardar comentario");
      return r;
    },
    onSuccess: async (_, vars) => {
      setComentariosDraft((prev) => ({
        ...prev,
        [vars.paso]: ""
      }));
      await queryClient.invalidateQueries({
        queryKey: ["revision", "detalle", vars.proyectoId]
      });
    },
    onSettled: () => {
      setPasoGuardando(null);
    }
  });
  const mutationAprobar = useMutation({
    mutationFn: async (proyectoId) => {
      const r = await aprobarProyectoRevision(proyectoId, esAdmin ? comiteActivo : void 0);
      if (!r.success) throw new Error(r.error || "No se pudo aprobar");
      return r;
    },
    onSuccess: async (_, proyectoId) => {
      await queryClient.invalidateQueries({
        queryKey: ["revision", "detalle", proyectoId]
      });
      await queryClient.invalidateQueries({
        queryKey: ["revision", "proyectos"]
      });
    }
  });
  const pasosPermitidos = useMemo(() => {
    const steps = queryDetalle.data?.pasosPermitidos ?? [];
    return [...steps].sort((a, b) => a - b);
  }, [queryDetalle.data?.pasosPermitidos]);
  const comite = esAdmin ? comiteActivo : queryLista.data?.comite ?? queryDetalle.data?.comite;
  const aprobaciones = queryDetalle.data?.aprobaciones ?? [];
  const aprobacionesMap = useMemo(() => {
    const map = /* @__PURE__ */ new Map();
    for (const a of aprobaciones) map.set(a.comite, a);
    return map;
  }, [aprobaciones]);
  const comiteActualAprobado = comite != null && aprobacionesMap.get(comite)?.aprobado === 1;
  const proyectoAprobado = queryDetalle.data?.proyecto?.estado === "aprobado";
  const abrirProyecto = (id) => {
    navigate({
      to: "/revision",
      search: {
        proyecto: id,
        comite: esAdmin ? comiteActivo : void 0
      },
      replace: true
    });
  };
  const volverALista = () => {
    setComentariosDraft({});
    navigate({
      to: "/revision",
      search: {
        proyecto: void 0,
        comite: esAdmin ? comiteActivo : void 0
      },
      replace: true
    });
  };
  return /* @__PURE__ */ jsxs("div", { className: "container mx-auto p-8 max-w-7xl", children: [
    /* @__PURE__ */ jsxs("div", { className: "mb-8 space-y-2", children: [
      /* @__PURE__ */ jsx("h1", { className: "text-4xl font-bold tracking-tight", children: "Revisión" }),
      /* @__PURE__ */ jsx("p", { className: "text-lg text-muted-foreground", children: "Vista por comité con detalle estructurado y comentarios por paso." })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "mb-4 flex items-center gap-3", children: [
      /* @__PURE__ */ jsxs(Badge, { variant: "secondary", children: [
        "Comité ",
        comite ?? "—"
      ] }),
      esAdmin && /* @__PURE__ */ jsx("div", { className: "w-52", children: /* @__PURE__ */ jsxs(Select, { value: String(comiteActivo), onValueChange: (v) => {
        const next = Number(v);
        if (next === 1 || next === 2 || next === 3) {
          setComiteActivo(next);
          navigate({
            to: "/revision",
            search: {
              proyecto: proyectoSeleccionadoId ?? void 0,
              comite: next
            },
            replace: true
          });
        }
      }, children: [
        /* @__PURE__ */ jsx(SelectTrigger, { children: /* @__PURE__ */ jsx(SelectValue, { placeholder: "Seleccione comité" }) }),
        /* @__PURE__ */ jsxs(SelectContent, { children: [
          /* @__PURE__ */ jsx(SelectItem, { value: "1", children: "Comité 1" }),
          /* @__PURE__ */ jsx(SelectItem, { value: "2", children: "Comité 2" }),
          /* @__PURE__ */ jsx(SelectItem, { value: "3", children: "Comité 3" })
        ] })
      ] }) })
    ] }),
    !proyectoSeleccionadoId && /* @__PURE__ */ jsxs(Card, { children: [
      /* @__PURE__ */ jsxs(CardHeader, { children: [
        /* @__PURE__ */ jsxs(CardTitle, { className: "flex items-center gap-2", children: [
          /* @__PURE__ */ jsx(HugeiconsIcon, { icon: File02Icon, className: "size-5" }),
          "Proyectos para revisión"
        ] }),
        /* @__PURE__ */ jsx(CardDescription, { children: "Seleccione un proyecto para abrir su vista de revisión." })
      ] }),
      /* @__PURE__ */ jsxs(CardContent, { children: [
        queryLista.isLoading && /* @__PURE__ */ jsx("div", { className: "py-16 flex items-center justify-center", children: /* @__PURE__ */ jsx(Spinner, { className: "size-8" }) }),
        queryLista.isError && /* @__PURE__ */ jsx("p", { className: "text-destructive text-center py-8", children: queryLista.error instanceof Error ? queryLista.error.message : "No se pudo cargar la revisión" }),
        !queryLista.isLoading && !queryLista.isError && (queryLista.data?.proyectos ?? []).length === 0 && /* @__PURE__ */ jsx("p", { className: "text-sm text-muted-foreground", children: "No hay proyectos enviados por revisar." }),
        !queryLista.isLoading && !queryLista.isError && (queryLista.data?.proyectos ?? []).length > 0 && /* @__PURE__ */ jsxs(Table, { children: [
          /* @__PURE__ */ jsx(TableHeader, { children: /* @__PURE__ */ jsxs(TableRow, { children: [
            /* @__PURE__ */ jsx(TableHead, { children: "ID" }),
            /* @__PURE__ */ jsx(TableHead, { children: "Proponente" }),
            /* @__PURE__ */ jsx(TableHead, { children: "Estado" }),
            /* @__PURE__ */ jsx(TableHead, { children: "Actualización" }),
            /* @__PURE__ */ jsx(TableHead, { className: "text-right", children: "Acción" })
          ] }) }),
          /* @__PURE__ */ jsx(TableBody, { children: (queryLista.data?.proyectos ?? []).map((p) => /* @__PURE__ */ jsxs(TableRow, { children: [
            /* @__PURE__ */ jsx(TableCell, { className: "font-mono", children: p.id }),
            /* @__PURE__ */ jsx(TableCell, { children: p.proponente_nombre }),
            /* @__PURE__ */ jsx(TableCell, { children: /* @__PURE__ */ jsx(Badge, { variant: p.estado === "aprobado" ? "default" : "secondary", children: p.estado }) }),
            /* @__PURE__ */ jsx(TableCell, { className: "text-muted-foreground", children: format(new Date(p.fecha_actualizacion), "d 'de' MMM yyyy, HH:mm", {
              locale: es
            }) }),
            /* @__PURE__ */ jsx(TableCell, { className: "text-right", children: /* @__PURE__ */ jsx(Button, { type: "button", variant: "outline", size: "sm", onClick: () => abrirProyecto(p.id), children: "Revisar" }) })
          ] }, p.id)) })
        ] })
      ] })
    ] }),
    proyectoSeleccionadoId != null && /* @__PURE__ */ jsxs("div", { className: "space-y-6", children: [
      /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between", children: [
        /* @__PURE__ */ jsxs(Button, { type: "button", variant: "outline", onClick: volverALista, children: [
          /* @__PURE__ */ jsx(HugeiconsIcon, { icon: ArrowLeft01Icon, className: "size-4 mr-2" }),
          "Regresar a proyectos"
        ] }),
        /* @__PURE__ */ jsx(Link, { to: "/revision", search: {
          proyecto: void 0,
          comite: esAdmin ? comiteActivo : void 0
        }, className: "text-sm text-muted-foreground hover:underline", children: "Ver tabla completa" })
      ] }),
      queryDetalle.isLoading && /* @__PURE__ */ jsx(Card, { children: /* @__PURE__ */ jsx(CardContent, { className: "py-16 flex items-center justify-center", children: /* @__PURE__ */ jsx(Spinner, { className: "size-7" }) }) }),
      queryDetalle.isError && /* @__PURE__ */ jsx(Card, { children: /* @__PURE__ */ jsx(CardContent, { className: "py-8", children: /* @__PURE__ */ jsx("p", { className: "text-destructive", children: queryDetalle.error instanceof Error ? queryDetalle.error.message : "No se pudo cargar el detalle" }) }) }),
      !queryDetalle.isLoading && !queryDetalle.isError && queryDetalle.data && /* @__PURE__ */ jsxs(Card, { children: [
        /* @__PURE__ */ jsxs(CardHeader, { children: [
          /* @__PURE__ */ jsxs(CardTitle, { children: [
            "Proyecto #",
            queryDetalle.data.proyecto?.id
          ] }),
          /* @__PURE__ */ jsx(CardDescription, { children: "Revise cada paso asignado a su comité y registre sus observaciones." })
        ] }),
        /* @__PURE__ */ jsxs(CardContent, { className: "space-y-4", children: [
          /* @__PURE__ */ jsxs("div", { className: "flex flex-wrap gap-2", children: [
            [1, 2, 3].map((n) => {
              const aprobado = aprobacionesMap.get(n)?.aprobado === 1;
              return /* @__PURE__ */ jsxs(Badge, { variant: aprobado ? "default" : "secondary", children: [
                "Comité ",
                n,
                ": ",
                aprobado ? "Aprobado" : "Pendiente"
              ] }, n);
            }),
            proyectoAprobado && /* @__PURE__ */ jsx(Badge, { className: "bg-green-600 text-white hover:bg-green-600", children: "Proyecto aprobado" })
          ] }),
          /* @__PURE__ */ jsx("div", { className: "flex items-center justify-end", children: /* @__PURE__ */ jsxs(Button, { type: "button", disabled: comite == null || comiteActualAprobado || mutationAprobar.isPending, onClick: () => {
            const id = queryDetalle.data.proyecto?.id;
            if (!id) return;
            mutationAprobar.mutate(id);
          }, children: [
            /* @__PURE__ */ jsx(HugeiconsIcon, { icon: CheckmarkCircle02Icon, className: "size-4 mr-2" }),
            comiteActualAprobado ? `Comité ${comite} aprobado` : mutationAprobar.isPending ? "Aprobando…" : `Aprobar comité ${comite ?? ""}`
          ] }) })
        ] })
      ] }),
      !queryDetalle.isLoading && !queryDetalle.isError && queryDetalle.data && pasosPermitidos.map((paso) => {
        const detalle = queryDetalle.data;
        const nombrePaso = detalle.nombresPaso?.[paso] ?? `Paso ${paso + 1}`;
        const datos = detalle.datosPorPaso?.[paso];
        const proyectoIdActual = detalle.proyecto?.id;
        const tiposArchivo = TIPOS_ARCHIVO_POR_PASO[paso] ?? [];
        const archivosPaso = (detalle.archivos ?? []).filter((a) => tiposArchivo.includes(a.tipo_archivo));
        const comentariosPaso = (detalle.comentarios ?? []).filter((c) => c.paso === paso);
        const draft = comentariosDraft[paso] ?? "";
        return /* @__PURE__ */ jsxs(Card, { children: [
          /* @__PURE__ */ jsxs(CardHeader, { children: [
            /* @__PURE__ */ jsxs(CardTitle, { className: "text-base", children: [
              "Paso ",
              paso + 1,
              ": ",
              nombrePaso
            ] }),
            /* @__PURE__ */ jsx(CardDescription, { children: "Información estructurada y comentarios de revisión." })
          ] }),
          /* @__PURE__ */ jsxs(CardContent, { className: "space-y-4", children: [
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx(Label, { children: "Información del paso" }),
              /* @__PURE__ */ jsx(DataViewer, { data: datos })
            ] }),
            archivosPaso.length > 0 && proyectoIdActual != null && /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx(Label, { children: "Archivos del paso" }),
              archivosPaso.map((archivo) => /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between rounded-md border p-2", children: [
                /* @__PURE__ */ jsx("span", { className: "text-sm", children: archivo.nombre_original }),
                /* @__PURE__ */ jsx(Button, { variant: "outline", size: "sm", asChild: true, children: /* @__PURE__ */ jsx("a", { href: urlArchivoRevision(proyectoIdActual, archivo.tipo_archivo, esAdmin ? comiteActivo : void 0), target: "_blank", rel: "noreferrer", children: "Ver archivo" }) })
              ] }, archivo.id))
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsxs(Label, { className: "flex items-center gap-2", children: [
                /* @__PURE__ */ jsx(HugeiconsIcon, { icon: Comment01Icon, className: "size-4" }),
                "Comentarios del comité"
              ] }),
              comentariosPaso.length === 0 && /* @__PURE__ */ jsx("p", { className: "text-sm text-muted-foreground", children: "Aún no hay comentarios en este paso." }),
              comentariosPaso.map((c) => /* @__PURE__ */ jsxs("div", { className: "rounded-md border p-3", children: [
                /* @__PURE__ */ jsx("p", { className: "text-sm whitespace-pre-wrap", children: c.comentario }),
                /* @__PURE__ */ jsxs("p", { className: "mt-2 text-xs text-muted-foreground", children: [
                  c.autor_nombre,
                  " · ",
                  format(new Date(c.fecha_comentario), "d 'de' MMM yyyy, HH:mm", {
                    locale: es
                  })
                ] })
              ] }, c.id))
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx(Label, { htmlFor: `comentario-${paso}`, children: "Agregar comentario del paso" }),
              /* @__PURE__ */ jsx(Textarea, { id: `comentario-${paso}`, rows: 3, placeholder: "Escriba su observación de revisión...", value: draft, onChange: (e) => setComentariosDraft((prev) => ({
                ...prev,
                [paso]: e.target.value
              })) }),
              /* @__PURE__ */ jsx("div", { className: "flex justify-end", children: /* @__PURE__ */ jsxs(Button, { type: "button", disabled: !draft.trim() || mutationComentario.isPending, onClick: () => {
                if (!detalle.proyecto?.id) return;
                setPasoGuardando(paso);
                mutationComentario.mutate({
                  proyectoId: detalle.proyecto.id,
                  paso,
                  comentario: draft.trim()
                });
              }, children: [
                /* @__PURE__ */ jsx(HugeiconsIcon, { icon: Tick02Icon, className: "size-4 mr-2" }),
                mutationComentario.isPending && pasoGuardando === paso ? "Guardando…" : "Guardar comentario"
              ] }) })
            ] })
          ] })
        ] }, paso);
      })
    ] })
  ] });
}
function DataViewer({
  data
}) {
  if (data == null) {
    return /* @__PURE__ */ jsx("div", { className: "rounded-lg border bg-muted p-3 text-sm text-muted-foreground", children: "Sin información capturada." });
  }
  if (Array.isArray(data)) {
    if (data.length === 0) {
      return /* @__PURE__ */ jsx("div", { className: "rounded-lg border bg-muted p-3 text-sm text-muted-foreground", children: "Sin elementos." });
    }
    const sonObjetos = data.every((item) => item && typeof item === "object" && !Array.isArray(item));
    if (sonObjetos) {
      const filas = data;
      const columnas = Array.from(new Set(filas.flatMap((f) => Object.keys(f))));
      return /* @__PURE__ */ jsx("div", { className: "rounded-lg border border-border/50 bg-card overflow-auto", children: /* @__PURE__ */ jsxs(Table, { children: [
        /* @__PURE__ */ jsx(TableHeader, { children: /* @__PURE__ */ jsx(TableRow, { children: columnas.map((col) => /* @__PURE__ */ jsx(TableHead, { children: toLabel(col) }, col)) }) }),
        /* @__PURE__ */ jsx(TableBody, { children: filas.map((fila, idx) => /* @__PURE__ */ jsx(TableRow, { children: columnas.map((col) => /* @__PURE__ */ jsx(TableCell, { children: valueToString(fila[col]) }, `${idx}-${col}`)) }, idx)) })
      ] }) });
    }
    return /* @__PURE__ */ jsx("div", { className: "rounded-lg border border-border/50 bg-card p-3", children: /* @__PURE__ */ jsx("ul", { className: "list-disc pl-5 space-y-1 text-sm", children: data.map((item, idx) => /* @__PURE__ */ jsx("li", { children: valueToString(item) }, idx)) }) });
  }
  if (typeof data === "object") {
    const entries = Object.entries(data);
    return /* @__PURE__ */ jsx("div", { className: "rounded-lg border border-border/50 bg-card overflow-hidden", children: /* @__PURE__ */ jsx(Table, { children: /* @__PURE__ */ jsx(TableBody, { children: entries.map(([key, value]) => /* @__PURE__ */ jsxs(TableRow, { children: [
      /* @__PURE__ */ jsx(TableCell, { className: "font-medium w-64 align-top", children: toLabel(key) }),
      /* @__PURE__ */ jsx(TableCell, { children: renderValue(value) })
    ] }, key)) }) }) });
  }
  return /* @__PURE__ */ jsx("div", { className: "rounded-lg border border-border/50 bg-card p-3 text-sm", children: String(data) });
}
function renderValue(value) {
  if (value == null) return /* @__PURE__ */ jsx("span", { className: "text-muted-foreground", children: "—" });
  if (typeof value === "boolean") return value ? "Sí" : "No";
  if (typeof value === "string") return formatIfDateString(value);
  if (Array.isArray(value)) {
    if (value.length === 0) return /* @__PURE__ */ jsx("span", { className: "text-muted-foreground", children: "Sin elementos" });
    return /* @__PURE__ */ jsx(DataViewer, { data: value });
  }
  if (typeof value === "object") {
    return /* @__PURE__ */ jsx(DataViewer, { data: value });
  }
  return String(value);
}
function valueToString(value) {
  if (value == null) return "—";
  if (typeof value === "boolean") return value ? "Sí" : "No";
  if (typeof value === "string") return formatIfDateString(value);
  if (typeof value === "object") return JSON.stringify(value);
  return String(value);
}
function toLabel(key) {
  const limpio = key.replace(/paso\d+$/i, "");
  return limpio.replace(/([A-Z])/g, " $1").replace(/_/g, " ").replace(/\s+/g, " ").trim().replace(/^./, (m) => m.toUpperCase());
}
function formatIfDateString(value) {
  const v = value.trim();
  if (!v) return value;
  const isoLike = /^(\d{4}-\d{2}-\d{2})([T\s]\d{2}:\d{2}(:\d{2}(\.\d{1,3})?)?(Z|[+-]\d{2}:?\d{2})?)?$/.test(v);
  if (!isoLike) return value;
  const d = new Date(v);
  if (Number.isNaN(d.getTime())) return value;
  return format(d, "d 'de' MMM yyyy", {
    locale: es
  });
}
export {
  RevisionPage as component
};
